/** @file   GfxManager.cpp
 * @brief   Implementation of GfxManager class.
 * @version $Revision: 1.5 $
 * @date    $Date: 2006/08/13 10:50:22 $
 * @author  Tomi Lamminsaari
 */

#include "GfxManager.h"
#include "eng2d.h"
#include "www_assert.h"

using std::string;
using namespace eng2d;

namespace WeWantWar {

///
/// Constants, datatypes and static methods
/// ============================================================================

DataWarehouse* GfxManager::allGraphics = 0;
BITMAP* GfxManager::titlePic = 0;
BMFont* GfxManager::titleFont = 0;
Font* GfxManager::storyFont = 0;
Font* GfxManager::ammoFont = 0;
Font* GfxManager::objectiveFont = 0;



void GfxLoadProgressObserver::NotifyWarehouseCreation(
        int aCurrentStep, int aTotalSteps, const std::string& aCurrentFilename )
{
  LOG_MESSAGE( aCurrentFilename.c_str() );
  eng2d::ProgressIndicator indicator( aTotalSteps );
  indicator.position( 26,180 );
  indicator.size(380,5);
  indicator.setStep( aCurrentStep );
  indicator.redraw();
}




int GfxManager::init()
{
  LOG_MESSAGE( "    Beginning to load graphics..." );
  
  titlePic = load_bitmap( "gfx/loading2.bmp", 0 );
  if (titlePic == 0) {
    return -1;
  }
  blit( titlePic, screen, 0,0, 0,0, titlePic->w, titlePic->h );
  
  GfxLoadProgressObserver* progressObserver = new GfxLoadProgressObserver;
  
  int ret = KErrNone;
  try {
    allGraphics = new DataWarehouse;
    FileList filelist( "gfx/config/all_graphics.txt" );
    WarehouseFactory::CreateWarehouse( allGraphics, filelist, progressObserver );
    
  } catch ( xRuntimeError except ) {
    string message = "        Reason: " + except.iMessage;
    LOG_MESSAGE( message.c_str() );
    ret = -1;
    
  }
  delete progressObserver;  progressObserver = 0;
  if ( ret != 0 ) {
    return ret;
  }
  LOG_MESSAGE( "    Graphics loaded" );
  
  // Load fonts
  LOG_MESSAGE( "    Loading fonts..." );
  try {
    titleFont = new BMFont( "data/menufont.dat" );
    storyFont = new Font( "data/storyfont.dat" );
    ammoFont = new Font( "data/AmmoFont.dat" );
    objectiveFont = new Font( "data/ObjectiveFont.dat" );
  }
  catch ( xIoError ) {
    LOG_MESSAGE( "        Failed!" );
    ret = -1;
  }
  
  return ret;
}


void GfxManager::cleanup()
{
  delete allGraphics;   allGraphics = 0;
  delete titleFont;     titleFont = 0;
  delete storyFont;     storyFont = 0;
  delete ammoFont;      ammoFont = 0;
  if ( titlePic != 0 ) {
    destroy_bitmap( titlePic );
    titlePic = 0;
  }
}


GraphicsContainer* GfxManager::findGfxContainer( int aUid )
{
  if ( allGraphics == 0 ) {
    return 0;
  }
  return dynamic_cast<GraphicsContainer*>( allGraphics->Find(aUid) );
}


BITMAP* GfxManager::findBitmap( int aUid, int aIndex )
    throw ( xOutOfBounds, xLogicError )
{
  if ( allGraphics == 0 ) {
    return 0;
  }
  GraphicsContainer* item = dynamic_cast<GraphicsContainer*>( allGraphics->Find(aUid) );
  BITMAP* bitmap = 0;
  if ( item != 0 ) {
    bitmap = item->GetBitmap( aIndex ); 
  }
  return bitmap;
}


RLE_SPRITE* GfxManager::findRleSprite( int aUid, int aIndex )
    throw ( xOutOfBounds, xLogicError )
{
  if ( allGraphics == 0 ) {
    return 0;
  }
  GraphicsContainer* item = dynamic_cast<GraphicsContainer*>( allGraphics->Find(aUid) );
  RLE_SPRITE* sprite = 0;
  if ( item != 0 ) {
    sprite = item->GetRleSprite( aIndex );
  }
  return sprite;
}


DataWarehouse* GfxManager::getWarehouse()
{
  return allGraphics;
}



///
/// Constructors, destructor and operators
/// ============================================================================

/** Default constructor.
 */
GfxManager::GfxManager()
{
}



/** Destructor.
 */
GfxManager::~GfxManager()
{
}



};  // end of namespace
